<?php

/**
 * This is a super class the handles
 * HTML attributes and setting
 * 
 * @author John Paul de Guzman <jpdguzman@gmail.com>
 * @version 0.1
 * @package DataGrid
 * */
class Html {
	
	/**
	 * HTML Tag name
	 * 
	 * @var string
	 * */
	var $_tag;
	
	/**
	 * HTML Tag attributes
	 * The format would be: attribute_name = value
	 * 
	 * <code>
	 * // set width & height to 200
	 * // HTML Output: <tagName width="200" height="200"></tagName>
	 * $_attributes = array("width" => "200",
	 * 						"height" => "200");
	 * </code>
	 * 
	 * @var array
	 * */
	var $_attributes;
	
	/**
	 * HTML generated Code
	 * */
	var $_html;
	
	/** 
	 * HTML Tag ID
	 * 
	 * @var string
	 * */
	var $_id;
	
	/**
	 * HTML Tag Class
	 * 
	 * @var string
	 * */
	var $_class;
	
	/**
	 * If the class is set to short tag
	 * */
	var $__isShortTag;
	
	
	/**
	 * Default Constructor for HTML object
	 * 
	 * Example implementation
	 * <code>
	 * // attributes for the tag
	 * $attributes = array("width" => "200",
	 *					   "height" => "200",
	 *					   "bgcolor" => "#f4f5f6");
	 * // Create a new TR Tag Object
	 * // TR is a full tag
	 * $tr = new Html("tr", $attributes);
	 * $tr->open();   // out: <tr>
	 * $tr->close();  // out: </tr> 
	 * 
	 * // Create a new INPUT Tag Object
	 * $input = new Html("input", $attributes, true);
	 * $input->open(); // out: <input />
	 * $input->close(); // out: NONE
	 * </code>
	 * 
	 * @param string tagName
	 * @param array attributes
	 * */
	function Html($tagName, $attributes = array(), $isShortTag=false) {
		// set HTML tag name
		$this->_tag = $tagName;
		// set attributes array
		$this->_attributes = $attributes;
		$this->__isShortTag = $isShortTag;
		
		// generate the html on instantion
		// this process increase the compile and run time
		// of the system when the object runs.
		// this avoids multiple loops running, thus increase in initial performance.
		$this->_html = $this->printHtml();
	}
	
	/**
	 * Set Attributes for HTML tag
	 * 
	 * @param array attributes
	 * */
	function setAttributes($attributes) {
		// set HTML attributes
		$this->_attributes = $attributes;
		
		// force generate Html code.
		$this->printHtml(true);
	}	
	
	/**
	 * Outputs the generated HTML Code base on given parameters
	 * if __isShortTag is set then the output would be
	 * <tagname />
	 * closing tag may not be required.
	 * However, if __isShortTag is not set, the output would be
	 * <tagname>
	 * closing tag is required to close the specified full tag
	 * 
	 * This function also appends attributes to the tag: class, id, and other specified attributes
	 * 
	 * forceGenerate: is a flag that indicates that the method would
	 * regenerate the original generate HTML tag.
	 * if the flag is set to true, then the system would force generate the
	 * html code.
	 * 
	 * 
	 * @return string
	 * @param bool forceGenerate
	 * */
	function printHtml($forceGenerate = false) {
		// if _html variable is not set
		// then generate the HTML tag output string
		if(empty($this->_html) || $forceGenerate) {
			// create the HTML for output
			$_htmlTag = "<".$this->_tag." ";
			
			if($this->_attributes != null) {
				// loop through all the attributes
				foreach($this->_attributes as $attr => $val) {
					// make sure that the attributes are in lower case
					$_htmlTag .= " ".strtolower($attr)."=\"$val\" ";
				}
			}
			
			// if id is set to the attributes array
			if(empty($this->_attributes['id'])) {
				// if ID is set
				if(!empty($this->_id)) {
					$_htmlTag .= " id=\"".$this->_id."\" ";
				}	
			}
			
			
			// if Class is set 
			if(!empty($this->_class)) {
				$_htmlTag .= " class=\"".$this->_class."\" ";
			}
			
			// if it is set to short tag
			if($this->__isShortTag) {
				$_htmlTag .= " /";
			}
			
			// close HTML tag
			$_htmlTag .= ">";
			
			
			// set the object HTML attribute
			$this->_html = $_htmlTag;
		} else {
			// else then get the generated HTML code
			$_htmlTag = $this->_html;
		}
		
		// return HTML tag
		return $_htmlTag;
	}
	
	/**
	 * Open an HTML tag
	 * it is an interface to printHTML method
	 * if __isShortTag is set the html output would be:
	 * <tagname attributes=value />
	 * 
	 * for full tag: close tag is needed in full tag
	 * <tagname attributes=value >
	 * 
	 * @return string generated Html tag
	 * */
	function open() {
		return $this->printHtml();
	}
	
	/**
	 * Close Tag
	 * Open() method should be called before calling this method.
	 * If the __isShortTag is set to through the close tag won't
	 * display anything.  However, if the __isShortTag is not set
	 * then the close tag method would return a close tag string. 
	 * 
	 * @return closetag 
	 * */
	function close() {
		if($this->__isShortTag) {
			$close = "";
		} else {
			$close = "</".$this->_tag.">\n"; 
		}
		
		return $close;
	}
	/**
	 * Set HTML ID attribute
	 * 
	 * @param string Id
	 * */
	function setId($id) {
		$this->_id = $id;
		
		$this->printHtml(true);
	}
	
	/**
	 * Set HTML Class attribute
	 * 
	 * @param class
	 * */
	function setClass($class) {
		$this->_class = $class;
		$this->printHtml(true);
	}
	
}
?>
