<?php

/**
 * Data Grid
 * The Data Grid Class is a simple tabular representation of
 * any array data fed into it.
 * 
 * @author John Paul de Guzman <jpdguzman@gmail.com>
 * @version 0.1
 * @package DataGrid
 * */
class DataGrid extends Html{
	
	/**
	 * Data Grid Header
	 * Header Container Object 
	 * 
	 * @var DataGridHeaderContainer
	 * @uses DataGridHeaderContainer	
	 * */
	var $_header;
	
	/**
	 * Data
	 * Collection of Data Objects
	 * This is a transformation of raw data passed to the controls
	 * to the DataContainer object.
	 * 
	 * @var DataGridRow array
	 * @uses DataGridRow
	 * */
	var $_data = array();
	
	/**
	 * Paging Object
	 * 
	 * @var DataGridPaging
	 * @uses DataGridPaging
	 * */
	 var $_paging;
	 
	 /**
	  * Raw Data
	  * Array that would be use to populate the datagrid.
	  * The raw data must follow a format regarding data grid
	  * evaluation and display
	  * 
	  * Raw Data variable should be an associative array.  This array
	  * has a name dependecies on the defined header
	  * 
	  * Example:
	  * <code>
	  *     // Data Grid Header List
	  *     $h_newsId = new DataGridHeader("item_id");
	  *     $h_newsTitle = new DataGridHeader("item_name");
	  *     $h_newsIntro = new DataGridHeader("item_description", "Introduction", $artr);
	  * 
	  *     // the header name should be use in forming the raw data array
	  *     // raw data array sould be.
	  *     $rawData = array(
	  *                    array("item_id" => "1", "item_name" => "Item1", "item_description" => "Item Description 1");,
	  *                    array("item_id" => "2", "item_name" => "Item2", "item_description" => "Item Description 2");,
	  * 				   array("item_id" => "3", "item_name" => "Item3", "item_description" => "Item Description 3"); 
	  *                );
	  *     // Once the control evaluates this array, it would use the header name provided in the DataGridHeader
	  *     // object as a pointer to the raw data provided.
	  * </code> 
	  * 
	  * @var mixed
	  * */
	 var $__rawData;
	 
	 /** 
	  * Data Grid Configuration
	  * This array controls the Data Grid's output and functional configuration.
	  * Id's additional controls, attributes, header, ect
	  * can be added in the configuration.
	  * 
	  * An Associative array following the attribute=value structure
	  * There are list of parameters for the Data Grid configuration
	  * listed on the constructor Method
	  * 
	  * @var mixed
	  */
	 var $__dataGridConfig;
	 
	 /**
	  * Data Rpw Attributes are set attribute definition for <tr> tags
	  * for the data grid.  Attributes array uses the format
	  * for creating or assigning attributes to an object.
	  * 
	  * DataRowAttributes is an associative array
	  * 
	  * @var mixed
	  * */
	 var $__dataRowAttributes;
	 
	 /***
	  * Data Grid Container Attributes are set attributes for
	  * <td> tags for the datagrid.  This attributes are nested under the
	  * DataRow object.
	  * 
	  * This uses the same format as other attribute array 
	  * 
	  * @var mixed
	  */
	 var $__dataContainerAttributes;
	 
	 
	 
	/**
	 * Default Constructor for creating a DataGrid
	 * Data Grid Configuration can be implement and be customized in different ways
	 * 
	 * View component documation for complete Documentation and configuration for the datagrid
	 * The DataGrid controls has 2 parts, plus an additional control:
	 * 
	 * Required Controls:
	 * Data Grid Header: This header is displayed on top of the data grid, this header is also used
	 * in mapping data grid data and putting then into their respected Columns, rows.
	 * 
	 * Data Grid Content: This serves as the body of the Data Grid, this is where the data feed into the grid
	 * was displayed, the data grid is highly dependent on the data grid header.
	 * 
	 * Optional Controls:
	 * Paging Control: Adds paging control to the data grid
	 * 
	 * @param mixed dataGridConfig
	 * */
	function DataGrid($dataGridConfig = array()) {
		
		// HTML object
		$this->_tag = "table";
		// data Grid Configurations
		$this->__dataGridConfig = $dataGridConfig;
		
		// if attributes is set
		if(!is_null($dataGridConfig['attributes'])) {
			$this->_attributes = $dataGridConfig['attributes'];
		}
		// if header is set
		if(!is_null($dataGridConfig['header'])) {
			$this->_header = $dataGridConfig['header'];
		}
		// if data is set
		if(!is_null($dataGridConfig['data'])) {
			$this->__rawData = $dataGridConfig['data'];
		}

		// data grid attributes
		// used for data grid row
		if(!is_null($dataGridConfig['dataRowAttributes'])) {
			$this->__dataRowAttributes = $dataGridConfig['dataRowAttributes'];
		}
		
		// used for data Container
		if(!is_null($dataGridConfig['dataContainerAttributes'])) {
			$this->__dataContainerAttributes = $dataGridConfig['dataContainerAttributes'];
		}
	}
	
	/**
	 * Fill method populates the grid with the data provided in the constructor
	 * this fill method can be called if you want to create a quick data grid display.
	 * This method can be override or not be performed if one wants to have a customize
	 * data grid control.  The fill method is a template for populating the grid with data.
	 * 
	 * This method only assumes that there are no additional controls to be added.
	 * This is the basic implementation of the Data Grid control.  For added controls or
	 * customize way, see documentation. 
	 *  
	 * @return void
	 * */
	function fill() {
		
		// get header names
		$headerNames = $this->_header->getAllHeaders();
		
		// loop through all loaded data
		foreach($this->__rawData as $data) {
			// create a grid row
			$row = new DataGridRow($this->__dataRowAttributes);
			
			// perform mapping of the header name to the raw data
			// read raw data code documentation
			foreach($headerNames as $hname => $header) {
				// create a data container
				$dr_elem = new DataContainer($data[$hname], $this->__dataContainerAttributes);
				
				// add element to row
				$row->addElement($dr_elem);
			}
			
			// add row to datagrid
			$this->addRow($row);  
		}
	}
	
	
	/** 
	 * Set Header object to the Datagrid
	 * This method can be used for customize controls over the datagrid
	 * 
	 * @see DataGridHeader, DataGridHeaderContainer
	 * @param DataGridHeaderContainer header
	 * */
	function setHeader($header) {
		$this->_header = $header;
	}
	
	/**
	 * Add New Row into the Data Grid
	 * Sets a default Grid ID to the row: dg_trid
	 * 
	 * For customize controls, id can be set via attributes
	 * array parameter of DataGridRow
	 * or use setId method.
	 * 
	 * <code>
	 *  // Implementation 1:
	 *  $attributes = array("id" => "datagrid_trid");
	 * 	$row = new DataGridRow($attributes);
	 *  
	 *  // Implementation 2:
	 *  $row = new DataGridRow();
	 *  $row->setId("datagrid_trid");
	 *  
	 *  // both produces the same output
	 *  // outputs: <tr id="datagrid_trid"></tr>
	 * </code>
	 * 
	 * Note if attribute "id" was set to attributes array, and then
	 * use the setId method, there might be problems regarding output
	 * the object would return:
	 * 
	 * <code>
	 *   $attributes = array("id" => "datagrid_trid");
	 * 	 $row = new DataGridRow($attributes);
	 *   $row->setId("datagrid_trid");
	 * 	 // outputs: <tr id="datagrid_trid" id="datagrid_trid"></tr>
	 * </code>
	 * 
	 * Make sure to set the id once.  The same rule applies to "class" attribute
	 * 
	 * @param DataGridRow row
	 * */
	function addRow($row) {
		// if the data Grid ID is blank
		if($row->_id == "" ) {
			$row->setId("dg_trid");
		}
		
		array_push($this->_data, $row);
	}
	
	/**
	 * Display the Data Grid
	 * If paging is set to true
	 * then there would be a paging controls printed below the control.
	 * 
	 * Override to create additional controls and custom output for data grid
	 * 
	 * @return string
	 * */
	function output() {
		///////////////////////////////////////////////////////////////
		///    INITIALIZATION
		///////////////////////////////////////////////////////////////
		// if paging is set
		$_isPaging = $this->__dataGridConfig['paging'];
		// paging location
		$_pagingLocation = $this->__dataGridConfig['pagingLocation'];
		// items to show: display items per page default is 10
		$_itemsToShow = $this->__dataGridConfig['itemstoshow'];
		
		// display header
		$_isDisplayHeader = $this->__dataGridConfig['displayHeader'];
		
		///////////////////////////////////////////////////////////////
		///    POPULATION OF DATA AND PAGING CONTROLS
		///////////////////////////////////////////////////////////////
		
		// if paging location is not set then 
		if($_pagingLocation == "") {
			// set paging location to bottom
			$_pagingLocation = "bottom";
		}
		
		// Populate container with data
		// initial data collection with would paging
		$dataCollection = $this->_data;
		// paging object
		$this->setPaging($_isPaging, $_itemsToShow, $paging = null, $dataCollection);
		
		$pagingAttributes = array("colspan" => count($this->_header->getAllHeaders()));
		
		// PAGING STYLE DEFINTION
		// Inline paging Row <tr>
		$inlinePaging = new Html("tr");
		// set Paging TR id
		if(!empty($this->__dataGridConfig['pagingTrId'])) {
			$inlinePaging->setId($this->__dataGridConfig['pagingTrId']);
		} 
		// set Paging TR class
		if(!empty($this->__dataGridConfig['pagingTrClass'])) {
			$inlinePaging->setClass($this->__dataGridConfig['pagingTrClass']);
		} 
		// set Paging TR Attributes
		if(!empty($this->__dataGridConfig['pagingTrAttributes'])) {
			$inlinePaging->setAttributes(array_merge($inlinePaging->_attributes, $this->__dataGridConfig['pagingTrAttributes']));
		}
		
		// Inline Paging Row Container <td>
		$pgTd = new Html("td", $pagingAttributes);
		// set Paging TD Id
		if(!empty($this->__dataGridConfig['pagingTdId'])) {
			$pgTd->setId($this->__dataGridConfig['pagingTdId']);
		}
		// set paging TD Class
		if(!empty($this->__dataGridConfig['pagingTdClass'])) {
			$pgTd->setClass($this->__dataGridConfig['pagingTdClass']);
		}
		// set paging TD Attributes
		if(!empty($this->__dataGridConfig['pagingTdAttributes'])) {
			$pgTd->setAttributes(array_merge($pgTd->_attributes, $this->__dataGridConfig['pagingTdAttributes']));
		}
		

		///////////////////////////////////////////////////////////////
		///    PERFORM DISPLAY & OUTPUT
		///////////////////////////////////////////////////////////////
		
		// DATA GRID: Open
		$_html .= $this->open();
		
		
		if($_pagingLocation == "top" || $_pagingLocation == "both") {
			$this->displayInlinePaging($inlinePaging, $pgTd, $_isPaging, $paging, $_html);
		}		
		
		// DATA GRID: Header
		$this->displayHeader($_isDisplayHeader, $_html);
		// DATA GRID: Data
		$this->displayGrid($this->__dataGridConfig['alternateType'], $dataCollection, $_html);		


		// inline paging display
		if($_pagingLocation == "bottom" || $_pagingLocation == "both") {
			$this->displayInlinePaging($inlinePaging, $pgTd, $_isPaging, $paging, $_html);
		}
		// DATA GRID: Close
		$_html .= $this->close();
			
		return $_html;
	}
	
	
	/* ------------------------------------------------------------
	 * 	DATA GRID DISPLAY COMPONENTS
	 *  These components can use reuse in creating a customize
	 *  datagrid view.
	 * ------------------------------------------------------------ */
	 
	 /**
	  * Display Inline Paging Component
	  * This component displays the paging inside the table intead of 
	  * outside the table or container
	  * 
	  * @param Html inlinePaging <tr>
	  * @param Html pgTd <td>
	  * @param bool _isPaging
	  * @param bool _pagingLocation
	  * @param DataGridPaging paging
	  * @param string _html
	  * */
	  function displayInlinePaging(&$inlinePaging, &$pgTd, $_isPaging, &$paging, &$_html) {
	  		$_html .= $inlinePaging->open();
			$_html .= $pgTd->open();
			
			// PAGING: Top
			$this->displayPaging($_isPaging, $paging, $_html);
			$_html .= $pgTd->close();
			$_html .= $inlinePaging->close();
	  }
	  
	  
	 /**
	  * Displays Header Data Grid Component
	  * if isDisplayHeader is set to false the item would not be displayed
	  * 
	  * @param bool _isDisplayHeader
	  * @param string _html
	  * */
	 function displayHeader($_isDisplayHeader, &$_html) {
	 	// if display header is set to true
	 	if($_isDisplayHeader === true || $_isDisplayHeader == "") {
	 		// display header
			$_html .= $this->_header->output();
		}
	 }
	 
	 function displayGrid($_switch, &$dataCollection, &$_html) {
	 	if($_switch == "" || $_switch == "default") { // Display Default
	 		$this->displayDefaultGrid($dataCollection, $_html);
	 	} 
	 	if($_switch == "id") { // Display Alternating Id
	 		// get the alternate ID Defintion
	 		$__id1 = $this->__dataGridConfig['alternateId1'];
	 		$__id2 = $this->__dataGridConfig['alternateId2'];
	 		
	 		$this->displayGridWithAlternatingId($__id1, $__id2, $dataCollection, $_html);
	 	} 
	 	if($_switch == "class") { // Display Alternating Class
	 		$__class1 = $this->__dataGridConfig['alternateClass1'];
	 		$__class2 = $this->__dataGridConfig['alternateClass2'];
	 		
	 		$this->displayGridWithAlternatingClass($__class1, $__class2, $dataCollection, $_html);
	 	} 
	 	if($_switch == "attributes") { // Display Alternating Attributes
	 		$__attr1 = $this->__dataGridConfig['alternateAttribute1'];
	 		$__attr2 = $this->__dataGridConfig['alternateAttribute2'];
	 		
	 		$this->displayGridWithAlternateAttributes($__attr1, $__attr2, $dataCollection, $_html);
	 	}
	 }
	 
	 /**
	  * Displays Default Data Grid Component
	  * This method displays the grid base on the populated data with out any additonal style defintion
	  * 
	  * 
	  * @param DataGridRow dataCollection
	  * @param string _html
	  * */
	 function displayDefaultGrid(&$dataCollection, &$_html) {
		// display the data
		foreach($dataCollection as $data) {
			// display the grid
			$_html .= $data->output();
		}
	 }
	 
	 /**
	  * Display Grid with Alternating ID Defintion
	  * 
	  * @param string id1
	  * @param string id2
	  * @param DataGridRow dataCollection
	  * @param string _html 
	  * */
	 function displayGridWithAlternatingId($id1, $id2, &$dataCollection, &$_html) {
	 	$this->_alternate("id", $id1, $id2, $dataCollection, $_html);
	 }
	 
	  /**
	  * Display Grid with Alternating Class Defintion
	  * 
	  * @param string class1
	  * @param string class2
	  * @param DataGridRow dataCollection
	  * @param string _html 
	  * */
	 function displayGridWithAlternatingClass($class1, $class2, &$dataCollection, &$_html) {
	 	$this->_alternate("class", $class1, $class2, $dataCollection, $_html);
	 }
	 
	 /**
	  * Display Grid with Alternating Attribute Array Definition
	  * 
	  * @param array styleDefintion1
	  * @param array styleDefintion2
	  * @param DataGridRow dataCollection
	  * @param string _html
	  * */
	 function displayGridWithAlternateAttributes($styleDefinition1, $styleDefinition2, &$dataCollection, &$_html) {
	 	// perform alternets
	 	$this->_alternate("attributes", $styleDefinition1, $styleDefinition2, $dataCollection, $_html);
	 }
	 
	 /**
	  * Perform alternating algorithm on data collection
	  * This alternate method uses a key pointer to determine what modification
	  * to perform
	  * Modification types are:
	  *  - _type: [attributes|id|class]
	  * 
	  * "attributes" this type uses an attribute array for grid definition
	  * "id" this type uses id for style defintion
	  * "class" this type uses class for style defintion
	  * 
	  * This alternate methods directly manipulate the row defintion of the grid <tr>
	  * 
	  * @param string _type: modification type
	  * @param [string|mixed] styleDefintion1
	  * @param [string|mixed] styleDefintion2
	  * @param DataGridRow dataCollection
	  * @param string _html
	  * @access private
	  * */
	 function _alternate($_type, $styleDefinition1, $styleDefinition2, &$dataCollection, &$_html) {
	 	$dataCollectionKey = array_keys($dataCollection);
	 	 $totalKey = count($dataCollectionKey);
	 	 
	 	// loop through the collection keys
	 	 for($i=0; $i<$totalKey; $i++) {
	 	 	//echo $dataCollectionKey[$i];
	 	 	$dataRow = $dataCollection[$dataCollectionKey[$i]];
	 	 	// if even: apply style defintion 1
	 	 	if(($i%2) == 0) {
	 	 		$style = $styleDefinition1;
	 	 	} else {	// if odd: apply style defintion 2
	 	 		$style = $styleDefinition2;
	 	 	}
	 	 	
	 	 	
	 	 	/* Style Switch */
	 	 	// if _type is set to "ATTRIBUTES" 
	 	 	if($_type == "attributes") {
 	 			$dataRow->setAttributes($style);
 	 		} 
 	 		// if _type is set to "ID"
 	 		elseif($_type == "id") {
 	 			$dataRow->setId($style);
 	 		} 
 	 		// if _type is set to "CLASS"
 	 		elseif($_type == "class") {
 	 			$dataRow->setClass($style);
 	 		}
	 	 		
	 	 	// output	
	 	 	$_html .= $dataRow->output();
	 	 }
	 }
	 
	 
	 /**
	  * Display Paging Component outputs the paging control
	  * for the datagrid
	  * The component contains 4 essentail element:
	  * if the controls should 
	  *  - perform paging,
	  *  - the paging location
	  * 
	  *  - the paging object
	  *  - html string
	  * 
	  * @param boolean _isPaging 
	  * @param DataGridPaging paging
	  * @param string _html
	  * */
	 function displayPaging($_isPaging,&$paging, &$_html) {
	 	if($_isPaging == "true") {
			$_html .= $paging->output();
				
		}
	 }
	 
	 function setPaging($_isPaging, $_itemsToShow, &$paging, &$dataCollection) {
	 	// if paging is set to true
		if($_isPaging == "true") {
			// if items to show is zero or not set
			if($_itemsToShow <= 0) {
				// set to items to show to 10
				$_itemsToShow = 10;
			}
			
			// create a DataGridPaging object
			$paging = new DataGridPaging($this->_data, $_itemsToShow);
			
			// set paging object
			$this->_paging = $paging;
			
			// create paging collection
			$dataCollection = $paging->performPaging();
		}
	 }
	 
}

/**
 * Data Container serves as the basic unit of the Data Grid
 * it serves as a container of any data element that has to be viewed or
 * displayed into the data grid.
 * 
 * 
 * @author John Paul de Guzman <jpdguzman@gmail.com>
 * @version 0.1
 * @package DataGrid
 * */
class DataContainer extends Html{
	
	/**
	 * Value of the container
	 * 
	 * @var string
	 */
	var $_dataElement;
	
	/**
	 * Default Constructor for Data Container
	 * 
	 * @param string data Element
	 * @param array attributes
	 * */
	function DataContainer($dataElement, $attributes = array()) {
		$this->_dataElement = $dataElement;
		
		// html element
		$this->_tag = "td";
		$this->_attributes = $attributes;
	}
	
	/**
	 * Output method for Data Container
	 * 
	 * @return string
	 * */
	function output() {
		$_html = $this->open();
		$_html .= $this->_dataElement;
		$_html .= $this->close();
		
		// html output 
		return $_html;
	}
}

/**
 * Handles the Collection of DataContainer and then form
 * a row
 * 
 * @author John Paul de Guzman <jpdguzman@gmail.com>
 * @version 0.1
 * @package DataGrid
 * */
class DataGridRow extends Html{
	
	/**
	 * Data Elements Collection
	 * 
	 * @var array
	 * */
	var $dataElements;
	
	/**
	 * Default Constructor
	 * 
	 * @param array attributes
	 * */
	function DataGridRow($attributes = array()) {
		// Html objects
		$this->_tag = "tr";
		$this->_attributes = $attributes;
		$this->dataElements = array();
	}
	
	/**
	 * Add Element to row
	 * 
	 * @param DataContainer element
	 * */
	function addElement($__element) {
		$_tempDataElements = array();
		$_tempDataElements = $this->dataElements;
		
		
		// if id is null
		if($__element->_id == "") {
			$__element->setId("dg_tdid");
		}
		
		array_push($_tempDataElements, $__element);
		$this->dataElements = $_tempDataElements;
	}
	
	/**
	 * Displays the Data Grid Row
	 * 
	 * @return string
	 * */
	function output() {
		$_html = $this->open();
		
		// if there are data elements found
		if(count($this->dataElements) > 0) {
			// loop through all the elements
			foreach($this->dataElements as $elements) {
				$_html .= $elements->output()."\n";
			}
		}
		// close tag
		$_html .= $this->close();
		
		// return Html code
		return $_html;
	}
	
}

/**
 * Data Grid Paging Class
 * Handles paging for the Datagrid
 * this uses the data collection to perform paging
 * to make the class independent and highly reusable
 * 
 * @author John Paul de Guzman <jpdguzman@gmail.com>
 * @version 0.1
 * @package DataGrid
 * */
class DataGridPaging {
	
	/**
	 * Items to show or to be displayed
	 * per page
	 * 
	 * @var int
	 * */
	var $__itemsToShow;
	
	/**
	 * Collection of items, data to be paged
	 * 
	 * @var mixed
	 * */
	var $__itemsCollection;
	
	/**
	 * Default Constructor for Data Grid paging
	 * 
	 * @param int itemsToShow
	 * @param mixed itemsCollection
	 * */
	function DataGridPaging($itemsCollection, $itemsToShow = '10') {
		// items to show per page
		$this->__itemsToShow = $itemsToShow;
		// items collection
		$this->__itemsCollection = $itemsCollection;
		
	}
	
	/**
	 * Displays paging
	 * 
	 * @return string
	 * */
	function output() {
		$pages = $this->getTotalNumberOfPages();
		
		// paging
		$_paging = "";
		
		foreach($pages as $page => $link) {
			$lnkAtr = array("href" => $_SERVER['PHP_SELF']."?".$link);
			$pagelnk = new Html("a", $lnkAtr);
			$_paging .= $pagelnk->open().$page.$pagelnk->close()." ";
		}
		return $_paging;
	}
	
	/**
	 * Perform Paging method
	 * This method uses itemsCollection to page and to collate items
	 * to perform paging
	 * 
	 * @return pages
	 * */	
	function performPaging() {
	  $total = count($this->__itemsCollection);
	 
	  // page offset
	  $page = $_GET['page'];
	  if($page == "") {
	  	$page = 1;
	  }
	  // page limits: page per display
	  $limit = $this->__itemsToShow;
	  
	  // number of pages
	  $numberOfPages = ceil($total / $limit);
	  // offset
	  $offset = ($page - 1)* $limit;
	  
	  // get all the keys of the elements
	  $itemKeys = array_keys($this->__itemsCollection);
	 
	  $items = array();
	  // get all page in the defined offest and page limits
	  for($i = $offset; $i<$limit*$page; $i++) {
	  	// put all items retrived in the db
	  	$keys = $itemKeys[$i];
	  	// prevent array out of bound
	  	if($this->__itemsCollection[$keys] != "") {
	  		// add element to the items array
	  		$items[$keys] = $this->__itemsCollection[$keys];
	  	}
	  }
	  
	  return $items;
	}
	
	/**
	 * Returns the total number of pages, with the links associated with the page
	 * 
	 * @return array page
	 * */
	function getTotalNumberOfPages() {
		$total = count($this->__itemsCollection);
		 
		// page offset
		$page = $_GET['page'];
		if($page == "") {
	  	    $page = 1;
	  	} 
		// page limits: page per display
		$limit = $this->__itemsToShow;
		  
		// number of pages
		$numberOfPages = ceil($total / $limit);
		
		$pages = array();
		
		// get the querystring
		$qstr = $_SERVER['QUERY_STRING'];
		
		// explode the querystring
		$qryStrElements = explode("&", $qstr);
		
		// loop through the querystring elements
		foreach($qryStrElements as $elementKey => $element) {
			// get each value pair and explode each elements
			$valuePair = explode("=", $element);
			
			// if there is a page item then remove that element
			if($valuePair[0] == "page") {
				// remove the element from the array
				unset($qryStrElements[$elementKey]);
			}
		}
		
		// perform rebuilding of query string
		$newQueryString = "";
		// loop through the querystring elements
		foreach($qryStrElements as $elementKey => $element) {
			$newQueryString .= $element."&";
		}
		
		// the new query string
		$newQueryString = substr($newQueryString, 0, strlen($newQueryString)-1);
		
	  	for($i=0; $i<$numberOfPages; $i++){
	  		$pages[$i+1] = $newQueryString."&page=".($i+1);
	  	}
	  	
	  	return $pages;
	}
}
?>
